﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using UnityEngine;

//The L-System class represents the base of an L-System, with its axiom value, rules and drawer for output
public class LSystem : MonoBehaviour
{
    //Public properties
    public string Axiom;
    public LSystemRule[] Rules;
    public LSystemDrawer Drawer;
    public string FriendlyName = "City";
    public int MinimumIterations = 1;
    public int MaximumIterations = 100;
    public int IterationsClamp;

    public string GeneratedSentence { get; private set; } = string.Empty;

    /// <summary>
    /// Generates a string sentence for this L-System, based on the rules and iterations
    /// </summary>
    /// <returns>The generated sentence</returns>
    public string Generate()
    {
        try
        {
            //Safety checks for clamping
            if(IterationsClamp < MinimumIterations)
            {
                IterationsClamp = MinimumIterations;
            }

            else if(IterationsClamp > MaximumIterations)
            {
                IterationsClamp = MaximumIterations;
            }

            //Compute and return our generated sentence using recursion
            GeneratedSentence = Iterate(Axiom, 0);
            return GeneratedSentence;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception during L-System generation. The exception is: " + ex);
            return string.Empty;
        }
    }

    /// <summary>
    /// Utilizes the drawer to generate and draw the L-System out
    /// </summary>
    /// <returns></returns>
    public bool Draw()
    {
        //Generate our sentence
        string generatedSentence = Generate();

        if (string.IsNullOrEmpty(generatedSentence))
        {
            return false;   //Something went wrong, abort
        }

        else
        {
            GeneratedSentence = generatedSentence;  //All is good, store it publically
        }

        return Drawer.Draw(GeneratedSentence);  //Now draw
    }

    /// <summary>
    /// Iterate utilizes the sentence in its current state and the number of iterations to continue applying the rules to a new result sentence
    /// </summary>
    /// <param name="sentence">The sentence in its current state</param>
    /// <param name="iterationsCount">How many iterations have been performed already?</param>
    /// <returns></returns>
    private string Iterate(string sentence, int iterationsCount)
    {
        if (iterationsCount >= IterationsClamp)
        {
            //We've gone deep enough, stop recursion
            return sentence;
        }

        else
        {
            StringBuilder sentenceBuilder = new StringBuilder();    //Builder of our new sentence

            //Loop through each character and apply the rules
            foreach (char character in sentence)
            {
                if(!ApplyRules(character, sentenceBuilder, iterationsCount))
                {
                    //Didn't apply rules, let's add it back on
                    sentenceBuilder.Append(character);
                }
            }

            return sentenceBuilder.ToString();  //Return the result
        }
    }

    /// <summary>
    /// Attempts to apply all rules to a character, using the sentence builder to append the output and an iterations count to know when to stop recursion
    /// </summary>
    /// <param name="character">The character to check</param>
    /// <param name="sentenceBuilder">The builder of the output sentence</param>
    /// <param name="iterationsCount">The number of iterations performed so far</param>
    /// <returns>Were any rules applied?</returns>
    private bool ApplyRules(char character, StringBuilder sentenceBuilder, int iterationsCount)
    {
        bool applied = false;

        //Loop through all the rules
        foreach (LSystemRule rule in Rules)
        {
            if (rule.Character == character)
            {
                //This rule applies, get the sentence from the rule
                string resultSentence = rule.ResultSentences[0];

                if(rule.SelectRandomResult)
                {
                    //We want a random sentence output, let's pick one
                    Randomizer.Regenerate();
                    int index = Randomizer.RNG.Next(0, rule.ResultSentences.Count);
                    resultSentence = rule.ResultSentences[index];
                }

                //Now get the new sentence from the one the rule provides and append it to our builder
                string newSentence = Iterate(resultSentence, iterationsCount + 1);
                sentenceBuilder.Append(newSentence);
                applied = true;
            }
        }

        return applied;
    }
}
